% Script to generating plots 5 (a-d)

% This file is part of Code and Data for the numerical experiments in 
% "Frequency-Based Reduced Models from Purely Time-Domain Data via Data Informativity"
% Copyright (C) 2023 Michael S. Ackermann
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
load Rand1000.mat
n_true = length(A);

rng(287346238);
T = 1000;
t_eval = 0:T;
U = randn(T+1,1);
Y = runDTSys(A,B,C,D,U,t_eval);

num = 400;
log_min_freq = -2; %lowest frequency/Ts wanted in frequency range
freqs = logspace(log_min_freq,log10(.99*pi),num);
z = exp(1i*freqs);


clear opts
opts.der_order = 1;
opts.num_windows = 20;
opts.num_windows_keep = 10;
opts.tau1 = 10^-10;
opts.tau2 = 10^-10;
opts.skip_condition_numbers = true;

%% Calculate frequency information from time domain data
tic
[Hz,nstd_Hz,cond_nums,residuals,opts] = CalculateTFVals(U,Y,z,opts);
toc
%% Calulate true frequency information
num = length(z);

I = eye(n_true);
H = @(s) C*((s*I-A)\B);
Hp = @(s) C*((s*I-A)\(-I*((s*I-A)\B)));
H_true = zeros(num,1);
Hp_true = zeros(num,1);
parfor i = 1:num
    H_true(i) = H(z(i));
    Hp_true(i) = Hp(z(i));
end


%% Calculate and report error
err = abs(Hz(:,1)-H_true);
err2 = norm(err);
err2rel = norm(err)/norm(H_true);

fprintf('Relative 2-norm error in TF estimates: %.5e\n',err2rel)
err_der = abs(Hz(:,2)-Hp_true);
err2D = norm(err_der);
err2relD = norm(err_der)/norm(Hp_true);
fprintf('Relative 2-norm error in Derivative TF estimates: %.5e\n',err2relD)

%% Plot recovered transfer function values
figure;
loglog(freqs,abs(H_true),'LineWidth',2)
hold on
loglog(freqs,abs(Hz(:,1)),'--','LineWidth',2)
legend('$|H(e^{\mathbf i \omega})|$',...
    '$|M_0(e^{\mathbf i \omega})|$','Interpreter',...
    'latex','Location','northwest')
xlim([10^(-2),pi])
ax = gca;
Default_TW = ax.TickLength;
Default_LW = ax.LineWidth;
ax.TickLength = Default_TW * 2;
ax.LineWidth = Default_LW * 2;
ax.FontSize = 16;
xlabel('$\omega$','Interpreter','latex','FontSize',20)
ylabel('Magnitude','Interpreter','latex','FontSize',20)
text(2e-2,3e0,'(a)','FontSize',30)

%% Plot recovered derivative values
figure;
loglog(freqs,abs(Hp_true),'LineWidth',2)
hold on
loglog(freqs,abs(Hz(:,2)),'--','LineWidth',2)
legend('$|H''(e^{\mathbf i \omega})|$',...
    '$|M_1(e^{\mathbf i \omega})|$','Interpreter',...
    'latex','Location','northwest')
xlim([10^(-2),pi])
ax = gca;
Default_TW = ax.TickLength;
Default_LW = ax.LineWidth;
ax.TickLength = Default_TW * 2;
ax.LineWidth = Default_LW * 2;
ax.FontSize = 16;
xlabel('$\omega$','Interpreter','latex','FontSize',20)
ylabel('Magnitude','Interpreter','latex','FontSize',20)
text(2e-2,5e1,'(c)','FontSize',30)

%% Plot error vs standard deviation for value
figure;
relerr = abs(Hz(:,1)-H_true)./abs(H_true);
loglog(freqs,relerr,'LineWidth',2)
hold on
loglog(freqs,nstd_Hz(:,1),'LineWidth',2)
legend('$\epsilon_{rel}$','$s_W$','Interpreter',...
    'latex','Location','northwest')
xlim([10^(-2),pi])
ylim([1e-15,1e-6])
ax = gca;
Default_TW = ax.TickLength;
Default_LW = ax.LineWidth;
ax.TickLength = Default_TW * 2;
ax.LineWidth = Default_LW * 2;
ax.FontSize = 16;
xlabel('$\omega$','Interpreter','latex','FontSize',20)
text(2e-2,2e-14,'(b)','FontSize',30)


%% Plot error vs standard deviation for derivative
figure;
relerrp = abs(Hz(:,2)-Hp_true)./abs(Hp_true);
loglog(freqs,relerrp,'LineWidth',2)
hold on
loglog(freqs,nstd_Hz(:,2),'LineWidth',2)
legend('$\epsilon_{rel}$','$s_W$','Interpreter',...
    'latex','Location','northwest')
xlim([10^(-2),pi])
ylim([1e-13,1e-6])
ax = gca;
Default_TW = ax.TickLength;
Default_LW = ax.LineWidth;
ax.TickLength = Default_TW * 2;
ax.LineWidth = Default_LW * 2;
ax.FontSize = 16;
xlabel('$\omega$','Interpreter','latex','FontSize',20)
text(2e-2,7e-13,'(d)','FontSize',30)
